<?php
require_once '../config/config.php';

if (!isAdmin()) {
    redirect('../dashboard.php');
}

$db = Database::getInstance()->getConnection();

// Handle actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'create') {
        $nama_depo = sanitize($_POST['nama_depo'] ?? '');
        $alamat = sanitize($_POST['alamat'] ?? '');
        $latitude = floatval($_POST['latitude'] ?? 0);
        $longitude = floatval($_POST['longitude'] ?? 0);
        $kapasitas_liter = intval($_POST['kapasitas_liter'] ?? 0);
        
        if (empty($nama_depo) || $latitude == 0 || $longitude == 0) {
            $_SESSION['error'] = 'Data wajib harus diisi';
        } else {
            $stmt = $db->prepare("INSERT INTO depo_pengiriman (nama_depo, alamat, latitude, longitude, kapasitas_liter) VALUES (?, ?, ?, ?, ?)");
            if ($stmt->execute([$nama_depo, $alamat, $latitude, $longitude, $kapasitas_liter])) {
                $_SESSION['success'] = 'Depo berhasil ditambahkan';
            } else {
                $_SESSION['error'] = 'Gagal menambahkan depo';
            }
        }
    } elseif ($action === 'update') {
        $id = intval($_POST['id'] ?? 0);
        $nama_depo = sanitize($_POST['nama_depo'] ?? '');
        $alamat = sanitize($_POST['alamat'] ?? '');
        $latitude = floatval($_POST['latitude'] ?? 0);
        $longitude = floatval($_POST['longitude'] ?? 0);
        $kapasitas_liter = intval($_POST['kapasitas_liter'] ?? 0);
        $is_active = intval($_POST['is_active'] ?? 1);
        
        $stmt = $db->prepare("UPDATE depo_pengiriman SET nama_depo = ?, alamat = ?, latitude = ?, longitude = ?, kapasitas_liter = ?, is_active = ? WHERE id = ?");
        if ($stmt->execute([$nama_depo, $alamat, $latitude, $longitude, $kapasitas_liter, $is_active, $id])) {
            $_SESSION['success'] = 'Depo berhasil diupdate';
        } else {
            $_SESSION['error'] = 'Gagal update depo';
        }
    } elseif ($action === 'delete') {
        $id = intval($_POST['id'] ?? 0);
        $stmt = $db->prepare("DELETE FROM depo_pengiriman WHERE id = ?");
        if ($stmt->execute([$id])) {
            $_SESSION['success'] = 'Depo berhasil dihapus';
        } else {
            $_SESSION['error'] = 'Gagal menghapus depo';
        }
    }
    
    redirect('depo.php');
}

// Get all depo
$stmt = $db->query("SELECT * FROM depo_pengiriman ORDER BY nama_depo");
$depo_list = $stmt->fetchAll();

?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Kelola Depo - <?php echo APP_NAME; ?></title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="https://unpkg.com/leaflet@1.9.4/dist/leaflet.css" />
    <script src="https://unpkg.com/leaflet@1.9.4/dist/leaflet.js"></script>
</head>
<body>
    <?php include '../includes/sidebar.php'; ?>
    
    <main class="main-content">
        <header class="content-header">
            <div class="header-left">
                <h1><i class="fas fa-warehouse"></i> Kelola Depo Pengiriman</h1>
                <p>Manajemen lokasi depo pengiriman air</p>
            </div>
            <button class="btn btn-primary" onclick="showDepoModal()">
                <i class="fas fa-plus"></i> Tambah Depo
            </button>
        </header>

        <?php if (isset($_SESSION['success'])): ?>
        <div class="alert alert-success">
            <i class="fas fa-check-circle"></i> <?php echo $_SESSION['success']; unset($_SESSION['success']); ?>
        </div>
        <?php endif; ?>

        <?php if (isset($_SESSION['error'])): ?>
        <div class="alert alert-danger">
            <i class="fas fa-exclamation-circle"></i> <?php echo $_SESSION['error']; unset($_SESSION['error']); ?>
        </div>
        <?php endif; ?>

        <div class="content-section">
            <div class="table-responsive">
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>Nama Depo</th>
                            <th>Alamat</th>
                            <th>Koordinat</th>
                            <th>Kapasitas (L)</th>
                            <th>Status</th>
                            <th>Aksi</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($depo_list)): ?>
                        <tr>
                            <td colspan="6" class="text-center">Belum ada depo</td>
                        </tr>
                        <?php else: ?>
                        <?php foreach ($depo_list as $depo): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($depo['nama_depo']); ?></td>
                            <td><?php echo htmlspecialchars($depo['alamat'] ?? '-'); ?></td>
                            <td>
                                <?php echo number_format($depo['latitude'], 6); ?>, <?php echo number_format($depo['longitude'], 6); ?>
                            </td>
                            <td><?php echo number_format($depo['kapasitas_liter']); ?></td>
                            <td>
                                <span class="badge badge-<?php echo $depo['is_active'] ? 'success' : 'danger'; ?>">
                                    <?php echo $depo['is_active'] ? 'Aktif' : 'Nonaktif'; ?>
                                </span>
                            </td>
                            <td>
                                <button class="btn btn-sm btn-primary" onclick="editDepo(<?php echo htmlspecialchars(json_encode($depo)); ?>)">
                                    <i class="fas fa-edit"></i>
                                </button>
                                <button class="btn btn-sm btn-danger" onclick="deleteDepo(<?php echo $depo['id']; ?>)">
                                    <i class="fas fa-trash"></i>
                                </button>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </main>

    <!-- Depo Modal -->
    <div id="depoModal" class="modal" style="display:none;">
        <div class="modal-content">
            <span class="close" onclick="document.getElementById('depoModal').style.display='none'">&times;</span>
            <h2 id="depoModalTitle">Tambah Depo</h2>
            <form method="POST" id="depoForm" class="form-container">
                <input type="hidden" name="action" value="create" id="depoAction">
                <input type="hidden" name="id" id="depo_id">
                <div class="form-group">
                    <label>Nama Depo</label>
                    <input type="text" name="nama_depo" id="nama_depo" required>
                </div>
                <div class="form-group">
                    <label>Alamat</label>
                    <textarea name="alamat" id="alamat" rows="3"></textarea>
                </div>
                <div class="form-group">
                    <label>Pilih Lokasi di Peta</label>
                    <div id="map" style="height: 300px; margin-top: 10px;"></div>
                    <small>Klik di peta untuk menentukan koordinat</small>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label>Latitude</label>
                        <input type="number" step="any" name="latitude" id="latitude" required>
                    </div>
                    <div class="form-group">
                        <label>Longitude</label>
                        <input type="number" step="any" name="longitude" id="longitude" required>
                    </div>
                </div>
                <div class="form-group">
                    <label>Kapasitas (Liter)</label>
                    <input type="number" name="kapasitas_liter" id="kapasitas_liter" min="0" value="0">
                </div>
                <div id="statusField" class="form-group" style="display:none;">
                    <label>Status</label>
                    <select name="is_active" id="is_active">
                        <option value="1">Aktif</option>
                        <option value="0">Nonaktif</option>
                    </select>
                </div>
                <button type="submit" class="btn btn-primary btn-block">Simpan</button>
            </form>
        </div>
    </div>

    <script src="../assets/js/main.js"></script>
    <script>
    let map, marker;
    
    function showDepoModal() {
        document.getElementById('depoModal').style.display = 'block';
        document.getElementById('depoForm').reset();
        document.getElementById('depo_id').value = '';
        document.getElementById('depoModalTitle').textContent = 'Tambah Depo';
        document.getElementById('depoAction').value = 'create';
        document.getElementById('statusField').style.display = 'none';
        
        setTimeout(() => {
            if (!map) {
                map = L.map('map').setView([-7.7956, 110.3695], 13);
                L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
                    attribution: '© OpenStreetMap contributors'
                }).addTo(map);
                
                map.on('click', function(e) {
                    const lat = e.latlng.lat;
                    const lng = e.latlng.lng;
                    document.getElementById('latitude').value = lat;
                    document.getElementById('longitude').value = lng;
                    
                    if (marker) {
                        map.removeLayer(marker);
                    }
                    marker = L.marker([lat, lng]).addTo(map);
                });
            }
        }, 100);
    }
    
    function editDepo(depo) {
        document.getElementById('depoModal').style.display = 'block';
        document.getElementById('depoModalTitle').textContent = 'Edit Depo';
        document.getElementById('depoAction').value = 'update';
        document.getElementById('depo_id').value = depo.id;
        document.getElementById('nama_depo').value = depo.nama_depo;
        document.getElementById('alamat').value = depo.alamat || '';
        document.getElementById('latitude').value = depo.latitude;
        document.getElementById('longitude').value = depo.longitude;
        document.getElementById('kapasitas_liter').value = depo.kapasitas_liter || 0;
        document.getElementById('is_active').value = depo.is_active;
        document.getElementById('statusField').style.display = 'block';
        
        setTimeout(() => {
            if (!map) {
                map = L.map('map').setView([depo.latitude, depo.longitude], 13);
                L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png').addTo(map);
                marker = L.marker([depo.latitude, depo.longitude]).addTo(map);
            } else {
                map.setView([depo.latitude, depo.longitude], 13);
                if (marker) map.removeLayer(marker);
                marker = L.marker([depo.latitude, depo.longitude]).addTo(map);
            }
            
            map.on('click', function(e) {
                const lat = e.latlng.lat;
                const lng = e.latlng.lng;
                document.getElementById('latitude').value = lat;
                document.getElementById('longitude').value = lng;
                if (marker) map.removeLayer(marker);
                marker = L.marker([lat, lng]).addTo(map);
            });
        }, 100);
    }
    
    function deleteDepo(id) {
        if (confirm('Hapus depo ini?')) {
            const form = document.createElement('form');
            form.method = 'POST';
            form.innerHTML = `
                <input type="hidden" name="action" value="delete">
                <input type="hidden" name="id" value="${id}">
            `;
            document.body.appendChild(form);
            form.submit();
        }
    }
    
    window.onclick = function(event) {
        if (event.target == document.getElementById('depoModal')) {
            document.getElementById('depoModal').style.display = 'none';
        }
    }
    </script>
    <style>
    .modal {
        display: none;
        position: fixed;
        z-index: 1000;
        left: 0;
        top: 0;
        width: 100%;
        height: 100%;
        background: rgba(0,0,0,0.5);
        overflow-y: auto;
    }
    
    .modal-content {
        background: white;
        margin: 3% auto;
        padding: 30px;
        border-radius: 15px;
        width: 90%;
        max-width: 800px;
    }
    
    .close {
        float: right;
        font-size: 28px;
        font-weight: bold;
        cursor: pointer;
        color: #999;
    }
    </style>
</body>
</html>

